package com.example.sefinsa_app.controllers;

import static androidx.constraintlayout.helper.widget.MotionEffect.TAG;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.util.Log;
import android.widget.Toast;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.JsonObjectRequest;
import com.example.sefinsa_app.api.API;
import com.example.sefinsa_app.migrations.DatabaseHelper;
import com.example.sefinsa_app.models.Folio;
import com.example.sefinsa_app.models.Poblacion;
import com.example.sefinsa_app.models.UsuarioFolio;
import com.example.sefinsa_app.utilities.ErrorChecker;
import com.example.sefinsa_app.utilities.VolleyS;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.sql.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


public class FolioController {

    private final DatabaseHelper databaseHelper;
    private final Context context;

    public FolioController(Context contexto) {
        this.databaseHelper = new DatabaseHelper(contexto);
        SQLiteDatabase db = databaseHelper.getWritableDatabase();
        context = contexto;

        //db.execSQL("DROP TABLE folios"); db.execSQL("DROP TABLE usuarios_folios");

        String createTableQuery = "CREATE TABLE IF NOT EXISTS folios (" +
                "id INTEGER PRIMARY KEY," +
                "fecha VARCHAR(10)," +
                "hora TIME," +
                "cliente_id VARCHAR(100)," +
                "ruta_id INTEGER," +
                "poblacion_id INTEGER," +
                "monto VARCHAR(50)," +
                "monto_letra VARCHAR(200)," +
                "concepto VARCHAR(1000)," +
                "firma_usuario VARCHAR(200000)," +
                "firma_cliente VARCHAR(200000)," +
                "sync VARCHAR(1)" +
                ");";

        db.execSQL(createTableQuery);

        String createUsuariosFoliosTable = "CREATE TABLE IF NOT EXISTS usuarios_folios (" +
                "usuario_id INTEGER," +
                "folio_id INTEGER" +
                ");";

        db.execSQL(createUsuariosFoliosTable);

        String createFoliosCanceladosTable = "CREATE TABLE IF NOT EXISTS folios_cancelados (" +
                "folio_id INTEGER," +
                "fecha VARCHAR(50)" +
                ")";

        db.execSQL(createFoliosCanceladosTable);
    }

    public interface FolioEliminarCallback {
        void onSuccess();
        void onError(String error);

        void onSucess();
    }

    public void getFoliosCanceladosServidor(FolioEliminarCallback callback) {
        VolleyS vs = new VolleyS(context);
        RequestQueue rq = vs.getRequestQueue();

        JSONObject data = new JSONObject();
        try {
            data.put("func", "getFoliosCancelados");
        } catch (JSONException e) {
            e.printStackTrace();
        }

        JsonObjectRequest req = new JsonObjectRequest(Request.Method.POST, API.urlPagos, data,
                new Response.Listener<JSONObject>() {
                    @Override
                    public void onResponse(JSONObject response) {
                        try {
                            JSONArray data = response.getJSONArray("data");
                            Log.d("Folios Cancelados", String.valueOf(data));
                            if (data.length() == 0) {
                                callback.onSuccess();
                            }
                            for (int i = 0; i < data.length(); i++) {
                                JSONObject folio = data.getJSONObject(i);
                                String folioId = folio.getString("folio_id");

                                SQLiteDatabase db = databaseHelper.getWritableDatabase();

                                String query = "DELETE FROM folios WHERE id = ?";
                                db.execSQL(query, new String[]{folioId});

                                Log.d("folioEliminado", folioId);

                                callback.onSuccess();
                            }
                        } catch (JSONException e) {
                            callback.onError(String.valueOf(e));
                            throw new RuntimeException(e);
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) { }
                });

        rq.add(req);
    }


    public double store(Folio folio) {
        if (folioExists(folio.getId())) {
            return -1;
        }

        SQLiteDatabase db = databaseHelper.getWritableDatabase();
        ContentValues cv = new ContentValues();

        cv.put("id", folio.getId());
        cv.put("fecha", folio.getFecha());
        cv.put("hora", folio.getHora());
        cv.put("cliente_id", folio.getCliente_id());
        cv.put("ruta_id", folio.getRuta_id());
        cv.put("poblacion_id", folio.getPoblacion_id());
        cv.put("monto", folio.getMonto());
        cv.put("monto_letra", folio.getMonto_letra());
        cv.put("concepto", folio.getConcepto());
        cv.put("firma_usuario", folio.getFirma_usuario());
        cv.put("firma_cliente", folio.getFirma_cliente());
        cv.put("sync", 0);

        String query = "DELETE FROM usuarios_folios WHERE folio_id = ?";
        db.execSQL(query, new String[]{folio.getId()});

        Log.d(TAG, "store: " + folio.getId());

        return db.insert("folios", null, cv);
    }

    public boolean folioExists(String folioId) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        Cursor cursor = db.rawQuery("SELECT id FROM folios WHERE id = ?", new String[]{folioId});
        boolean exists = cursor.moveToFirst();
        cursor.close();
        db.close();
        return exists;
    }


    public ArrayList<Folio> index() {
        ArrayList<Folio> folios = new ArrayList<>();
        SQLiteDatabase db = databaseHelper.getReadableDatabase();

        String query = "SELECT * FROM folios";
        Cursor cursor = db.rawQuery(query, null);

        if (cursor.moveToFirst()) {
            do {
                String id = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String colocadora_id = cursor.getString(6);
                String monto = cursor.getString(7);
                String monto_letra = cursor.getString(8);
                String concepto = cursor.getString(9);
                String firma_usuario = cursor.getString(10);
                String firma_cliente = cursor.getString(11);

                Folio folio = new Folio(
                        id,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        colocadora_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        return folios;
    }

    public Folio show(String id) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT * FROM folios WHERE id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{id});

        if (cursor.moveToFirst()) {
            String fecha = cursor.getString(1);
            String hora = cursor.getString(2);
            String cliente_id = cursor.getString(3);
            String ruta_id = cursor.getString(4);
            String poblacion_id = cursor.getString(5);
            String colocadora_id = cursor.getString(6);
            String monto = cursor.getString(7);
            String monto_letra = cursor.getString(8);
            String concepto = cursor.getString(9);
            String firma_usuario = cursor.getString(10);
            String firma_cliente = cursor.getString(11);

            Folio folio = new Folio(
                    id,
                    fecha,
                    hora,
                    cliente_id,
                    ruta_id,
                    poblacion_id,
                    colocadora_id,
                    monto,
                    monto_letra,
                    concepto,
                    firma_usuario,
                    firma_cliente
            );
            cursor.close();
            return folio;
        }
        cursor.close();
        return null;
    }

    public ArrayList<Folio> getLikeFolios(String id, ArrayList<String> rutas) {
        ArrayList<Folio> folios = new ArrayList<>();
        SQLiteDatabase db = databaseHelper.getReadableDatabase();

        StringBuilder query = new StringBuilder("SELECT * FROM folios WHERE id LIKE ? AND ruta_id IN (");
        for (int i = 0; i < rutas.size(); i++) {
            query.append(rutas.get(i));
            if (i < rutas.size() - 1) {
                query.append(",");
            }
        }
        query.append(") ORDER BY id DESC");
        Cursor cursor = db.rawQuery(String.valueOf(query), new String[]{"%" + id + "%"});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getLikeNombre(String id, String date, ArrayList<String> rutas) {
        ArrayList<Folio> folios = new ArrayList<>();
        SQLiteDatabase db = databaseHelper.getReadableDatabase();

        StringBuilder query = new StringBuilder("SELECT * FROM folios WHERE cliente_id LIKE ? AND fecha = ? AND ruta_id IN (");
        for (int i = 0; i < rutas.size(); i++) {
            query.append(rutas.get(i));
            if (i < rutas.size() - 1) {
                query.append(",");
            }
        }
        query.append(") ORDER BY id DESC");

        Cursor cursor = db.rawQuery(String.valueOf(query), new String[]{"%" + id + "%", date});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioCliente (String id) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE cliente_id = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{id});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioFechaRutaId (String id, ArrayList<String> rutas) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        StringBuilder query = new StringBuilder("SELECT * FROM folios WHERE fecha = ? AND ruta_id IN (");
        for (int i = 0; i < rutas.size(); i++) {
            query.append(rutas.get(i));
            if (i < rutas.size() - 1) {
                query.append(",");
            }
        }
        query.append(") ORDER BY id DESC");

        Log.d(TAG, "getFolioFechaRutaId: " + query);

        Cursor cursor = db.rawQuery(query.toString(), new String[]{id});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioClienteFecha (String id, String date) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE fecha = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{date, id});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioRuta (String id) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE ruta_id = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{id});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioRutaFecha (String id, String date) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE ruta_id = ? AND fecha = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{id, date});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioFechaRuta (String id, String date) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE ruta_id = ? AND fecha = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{id, date});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioPoblacion (String id) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE poblacion_id = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{id});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioPoblacionFecha (String id, String date) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE poblacion_id = ? AND fecha = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{id, date});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioFechaRutaPoblacion(String id, String date, String pid) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE ruta_id = ? AND fecha = ? AND poblacion_id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{id, date, pid});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioClienteRutaPoblacion (String cid, String rid, String pid) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE cliente_id = ? AND ruta_id = ? AND poblacion_id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{cid, rid, pid});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioClienteRutaPoblacionFecha (String cid, String rid, String pid, String date) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE cliente_id = ? AND ruta_id = ? AND poblacion_id = ? AND fecha = ?";
        Cursor cursor = db.rawQuery(query, new String[]{cid, rid, pid, date});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioRutaPoblacion (String rid, String pid) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE ruta_id = ? AND poblacion_id = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{rid, pid});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public ArrayList<Folio> getFolioRutaPoblacionFecha (String rid, String pid, String date) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE ruta_id = ? AND poblacion_id = ? AND fecha = ? ORDER BY id DESC";
        Cursor cursor = db.rawQuery(query, new String[]{rid, pid, date});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return folios;
    }

    public void recreate() {
        String q = "DROP TABLE folios";
        SQLiteDatabase wdb = databaseHelper.getWritableDatabase();
        wdb.execSQL(q);
        String createTableQuery = "CREATE TABLE IF NOT EXISTS folios (" +
                "id INTEGER PRIMARY KEY," +
                "fecha VARCHAR(10)," +
                "hora TIME," +
                "cliente_id VARCHAR(100)," +
                "ruta_id INTEGER," +
                "poblacion_id INTEGER," +
                "monto VARCHAR(50)," +
                "monto_letra VARCHAR(200)," +
                "concepto VARCHAR(1000)," +
                "firma_usuario VARCHAR(200000)," +
                "firma_cliente VARCHAR(200000)," +
                "sync VARCHAR(1)" +
                ");";

        wdb.execSQL(createTableQuery);
        wdb.close();
    }

    public ArrayList<Folio> getUnsyncedFolios() {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<Folio> folios = new ArrayList<>();
        String query = "SELECT * FROM folios WHERE sync = 0";
        Cursor cursor = db.rawQuery(query, new String[]{});

        if (cursor.moveToFirst()) {
            do {
                String nFolio = cursor.getString(0);
                String fecha = cursor.getString(1);
                String hora = cursor.getString(2);
                String cliente_id = cursor.getString(3);
                String ruta_id = cursor.getString(4);
                String poblacion_id = cursor.getString(5);
                String monto = cursor.getString(6);
                String monto_letra = cursor.getString(7);
                String concepto = cursor.getString(8);
                String firma_usuario = cursor.getString(9);
                String firma_cliente = cursor.getString(10);

                Folio folio = new Folio(
                        nFolio,
                        fecha,
                        hora,
                        cliente_id,
                        ruta_id,
                        poblacion_id,
                        monto,
                        monto_letra,
                        concepto,
                        firma_usuario,
                        firma_cliente
                );
                folios.add(folio);
            } while (cursor.moveToNext());

            cursor.close();
            db.close();
        }

        db.close();
        return folios;
    }

    public String getLastestFolioID () {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT id FROM folios ORDER BY id DESC LIMIT 1";
        Cursor cursor = db.rawQuery(query, new String[]{});

        String folio = "";
        if (cursor.moveToFirst()) {
            folio = cursor.getString(0);
        }

        cursor.close();
        db.close();

        return folio;
    }

    public void syncFolio(
            String folio,
            String fecha,
            String hora,
            String cliente_id,
            String ruta_id,
            String poblacion_id,
            String monto,
            String monto_letra,
            String concepto,
            String firma_usuario,
            String firma_cliente) {
        JSONObject data = new JSONObject();
        try {
            data.put("func", "storeFolio");
            data.put("folio", folio);
            data.put("fecha", fecha);
            data.put("hora", hora);
            data.put("cliente_id", cliente_id);
            data.put("ruta_id", ruta_id);
            data.put("poblacion_id", poblacion_id);
            data.put("monto", monto);
            data.put("monto_letra", monto_letra);
            data.put("concepto", concepto);
            data.put("firma_usuario", firma_usuario);
            data.put("firma_cliente", firma_cliente);
        } catch (JSONException e) { }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPagos, data,
                new Response.Listener<JSONObject>() {
                    @Override
                    public void onResponse(JSONObject response) {
                        Log.d("sync repsonse", "onResponse: " + response);
                        String query = "UPDATE folios SET sync = 1 WHERE id = ?";
                        SQLiteDatabase db = databaseHelper.getWritableDatabase();
                        db.execSQL(query, new String[]{folio});
                        db.close();
                    }

                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        Log.d("sync error", "onErrorResponse: " + error);
                    }
                });

        request.setRetryPolicy(new DefaultRetryPolicy(
                5000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        request.setShouldCache(false);

        VolleyS vs = VolleyS.getInstance(context);
        RequestQueue requestQueue = vs.getRequestQueue();

        requestQueue.add(request);
    }

    public interface FolioCountCallback {
        void onSuccess(int total);
        void onError(String error);
    }

    public String getPoblacionA(String id) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT nombre_poblacion FROM poblaciones WHERE id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{id});
        String poblacion = "";
        if (cursor.moveToFirst()) {
            poblacion = cursor.getString(0);
        }
        cursor.close();
        db.close();
        return poblacion;
    }

    public void getTotalFolios(FolioCountCallback callback) {
        JSONObject data = new JSONObject();
        try {
            data.put("func", "countFolios");
        } catch (JSONException e) { }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPagos, data,
                new Response.Listener<JSONObject>() {
                    @Override
                    public void onResponse(JSONObject response) {
                        try {
                            if (response.getString("status").equals("success")) {
                                JSONArray data = response.getJSONArray("data");
                                if (data.length() > 0) {
                                    JSONObject obj = data.getJSONObject(0);
                                    int total = obj.getInt("total");
                                    callback.onSuccess(total);
                                } else {
                                    callback.onError("No se encontró información.");
                                }
                            } else {
                                callback.onError("Error del servidor: " + response.getString("message"));
                            }
                        } catch (JSONException e) {
                            callback.onError("Error al parsear respuesta JSON.");
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        callback.onError("Error de red: " + error.getMessage());
                    }
                });

        request.setRetryPolicy(new DefaultRetryPolicy(
                5000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));
        request.setShouldCache(false);

        VolleyS.getInstance(context).getRequestQueue().add(request);
    }

    public Integer getFolioCount() {
        SQLiteDatabase db = new DatabaseHelper(context).getReadableDatabase();
        String query = "SELECT COUNT(*) FROM folios";
        Cursor cursor = db.rawQuery(query, null);
        cursor.moveToFirst();
        int count = cursor.getInt(0);
        cursor.close();
        db.close();
        return count;
    }

    public interface FolioDownloadCallback {
        void onDownloadComplete();
    }

    public void getFolioDataAndSave(FolioDownloadCallback callback) {
        JSONObject data = new JSONObject();
        try {
            data.put("func", "indexFolio");
        } catch (JSONException e) {
            e.printStackTrace();
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPagos, data,
                new Response.Listener<JSONObject>() {
                    @Override
                    public void onResponse(JSONObject response) {
                        try {
                            if (response.getString("status").equals("success")) {
                                JSONArray dataArray = response.getJSONArray("data");

                                for (int i = 0; i < dataArray.length(); i++) {
                                    JSONObject folioObject = dataArray.getJSONObject(i);
                                    Folio newFolio = new Folio(
                                            folioObject.getString("id"),
                                            folioObject.getString("fecha"),
                                            folioObject.getString("hora"),
                                            folioObject.getString("cliente_id"),
                                            folioObject.getString("ruta_id"),
                                            folioObject.getString("poblacion_id"),
                                            folioObject.getString("monto"),
                                            folioObject.getString("monto_letra"),
                                            folioObject.getString("concepto"),
                                            folioObject.getString("firma_usuario"),
                                            folioObject.getString("firma_cliente")
                                    );

                                    Log.d("folioDescargado", newFolio.toString());

                                    Folio existingFolio = getFolioFromDbById(newFolio.getId());

                                    if (existingFolio == null) {
                                        storeSync(newFolio);
                                    } else {
                                        if (!newFolio.equals(existingFolio)) {
                                            updateFolioInDb(newFolio);
                                            Log.d("folioActualizado", newFolio.toString());
                                        } else {
                                            Log.d("folioSinCambios", newFolio.getId());
                                        }
                                    }
                                }

                                callback.onDownloadComplete();
                            } else {
                                Toast.makeText(context, "Error del servidor: " + response.getString("message"), Toast.LENGTH_SHORT).show();
                            }
                        } catch (JSONException e) {
                            Toast.makeText(context, "Error al parsear respuesta JSON.", Toast.LENGTH_SHORT).show();
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        Toast.makeText(context, "Error de red: " + error.getMessage(), Toast.LENGTH_SHORT).show();
                    }
                });

        request.setRetryPolicy(new DefaultRetryPolicy(
                5000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));
        request.setShouldCache(false);

        VolleyS.getInstance(context).getRequestQueue().add(request);
    }

    private void updateFolioInDb(Folio newFolio) {
        SQLiteDatabase db = databaseHelper.getWritableDatabase();
        ContentValues cv = new ContentValues();
        cv.put("fecha", newFolio.getFecha());
        cv.put("hora", newFolio.getHora());
        cv.put("cliente_id", newFolio.getCliente_id());
        cv.put("ruta_id", newFolio.getRuta_id());
        cv.put("poblacion_id", newFolio.getPoblacion_id());
        cv.put("monto", newFolio.getMonto());
        cv.put("monto_letra", newFolio.getMonto_letra());
        cv.put("concepto", newFolio.getConcepto());
        cv.put("firma_usuario", newFolio.getFirma_usuario());
        cv.put("firma_cliente", newFolio.getFirma_cliente());
        cv.put("sync", 1);
        db.update("folios", cv, "id = ?", new String[]{newFolio.getId()});
        db.close();
    }

    private Folio getFolioFromDbById(String id) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT * FROM folios WHERE id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{id});
        Folio folio = null;

        if (cursor.moveToFirst()) {
            String nFolio = cursor.getString(0);
            String fecha = cursor.getString(1);
            String hora = cursor.getString(2);
            String cliente_id = cursor.getString(3);
            String ruta_id = cursor.getString(4);
            String poblacion_id = cursor.getString(5);
            String monto = cursor.getString(6);
            String monto_letra = cursor.getString(7);
            String concepto = cursor.getString(8);
            String firma_usuario = cursor.getString(9);
            String firma_cliente = cursor.getString(10);

            folio = new Folio(
                    nFolio,
                    fecha,
                    hora,
                    cliente_id,
                    ruta_id,
                    poblacion_id,
                    monto,
                    monto_letra,
                    concepto,
                    firma_usuario,
                    firma_cliente
            );
            cursor.close();
            db.close();
            return folio;
        }
        cursor.close();
        db.close();
        return folio;
    }

    public double storeSync(Folio folio) {
        if (folioExists(folio.getId())) {
            return -1;
        }

        SQLiteDatabase db = databaseHelper.getWritableDatabase();
        ContentValues cv = new ContentValues();

        cv.put("id", folio.getId());
        cv.put("fecha", folio.getFecha());
        cv.put("hora", folio.getHora());
        cv.put("cliente_id", folio.getCliente_id());
        cv.put("ruta_id", folio.getRuta_id());
        cv.put("poblacion_id", folio.getPoblacion_id());
        cv.put("monto", folio.getMonto());
        cv.put("monto_letra", folio.getMonto_letra());
        cv.put("concepto", folio.getConcepto());
        cv.put("firma_usuario", folio.getFirma_usuario());
        cv.put("firma_cliente", folio.getFirma_cliente());
        cv.put("sync", 1);

        String query = "DELETE FROM usuarios_folios WHERE folio_id = ?";
        db.execSQL(query, new String[]{folio.getId()});

        Log.d(TAG, "storeSync: " + folio.getId());

        return db.insert("folios", null, cv);
    }

    public void prepararFolios(String empleadoId, String cantidad) {
        JSONObject data = new JSONObject();
        try {
            data.put("func", "prepararFolios");
            data.put("usuario_id", empleadoId);
            data.put("cantidad", cantidad);
        } catch (JSONException e) { }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPagos, data,
                new Response.Listener<JSONObject>() {
                    @Override
                    public void onResponse(JSONObject response) {
                        try {
                            if (response.getString("status").equals("success")) {
                                JSONArray dataArray = response.getJSONArray("data");

                                for (int i = 0; i < dataArray.length(); i++) {
                                    JSONObject folioObject = dataArray.getJSONObject(i);
                                    UsuarioFolio usuarioFolio = new UsuarioFolio(
                                            folioObject.getString("usuario_id"),
                                            folioObject.getString("folio_id")
                                    );

                                    storeUsuarioFolio(usuarioFolio);
                                }
                            } else {
                                Toast.makeText(context, "Error del servidor: " + response.getString("message"), Toast.LENGTH_SHORT).show();
                            }
                        } catch (JSONException e) {
                            Toast.makeText(context, "Error al parsear respuesta JSON.", Toast.LENGTH_SHORT).show();
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        Log.d("sync error", "onErrorResponse: " + error);
                    }
                });

        request.setRetryPolicy(new DefaultRetryPolicy(
                5000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));
        request.setShouldCache(false);

        VolleyS.getInstance(context).getRequestQueue().add(request);
    }

    public void storeUsuarioFolio (UsuarioFolio usuarioFolio) {
        SQLiteDatabase db = databaseHelper.getWritableDatabase();
        ContentValues cv = new ContentValues();
        cv.put("usuario_id", usuarioFolio.getUsuarioId());
        cv.put("folio_id", usuarioFolio.getFolioId());

        db.insert("usuarios_folios", null, cv);
        db.close();
    }

    public ArrayList<UsuarioFolio> getUsuarioFolios(String usuarioId) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        ArrayList<UsuarioFolio> folios = new ArrayList<>();
        String query = "SELECT * FROM usuarios_folios WHERE usuario_id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{usuarioId});
        ArrayList<UsuarioFolio> usuarioFolios = new ArrayList<>();

        if (cursor.moveToFirst()) {
            do {
                String folioId = cursor.getString(1);
                usuarioFolios.add(new UsuarioFolio(folioId));
            } while (cursor.moveToNext());
        }

        cursor.close();
        db.close();

        return usuarioFolios;
    }

    public String getFirstAvailableFolioId (String empeladoId) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT folio_id FROM usuarios_folios WHERE usuario_id = ? ORDER BY folio_id ASC LIMIT 1";
        Cursor cursor = db.rawQuery(query, new String[]{empeladoId});

        String folioId = "";
        if (cursor.moveToFirst()) {
            folioId = cursor.getString(0);
        }

        cursor.close();
        db.close();

        return folioId;
    }

    public String getPoblacion (String id) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT nombre_poblacion FROM poblaciones WHERE id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{id});
        String poblacion = "";
        if (cursor.moveToFirst()) {
            poblacion = cursor.getString(0);
        }
        cursor.close();
        db.close();
        return poblacion;
    }

    public String verificarRuta (String txt) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT id FROM rutas WHERE nombre_ruta = ?";
        Cursor cursor = db.rawQuery(query, new String[]{txt});
        String ruta = "";
        if (cursor.moveToFirst()) {
            ruta = cursor.getString(0);
        }
        cursor.close();
        db.close();
        return ruta;
    }

    public String verificarPoblacion (String txt) {
        SQLiteDatabase db = databaseHelper.getReadableDatabase();
        String query = "SELECT id FROM poblaciones WHERE nombre_poblacion = ?";
        Cursor cursor = db.rawQuery(query, new String[]{txt});
        String poblacion = "";
        if (cursor.moveToFirst()) {
            poblacion = cursor.getString(0);
        }
        cursor.close();
        db.close();
        return poblacion;
    }
}